/*
 * Math3D.h
 *
 * Created 8/1/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/1/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef MATH3D_H
 #define MATH3D_H

 #include "Vector.h"

 #ifndef PI
 #define PI 3.14159f
 #endif // PI
 
 #ifndef ONE_EIGHTY
 #define ONE_EIGHTY 180.0f
 #endif // ONE_EIGHTY

 namespace Math3D
 {
    template <typename T> inline T degrees_to_radians( const T& degrees );
    template <typename T> inline T get_readjusted_angle_degrees( const T& degrees );
    template <typename T> inline T radians_to_degrees( const T& radians );
 }
 
 /**
  * degrees_to_radians() returns the radians given the degrees.
  *
  * @param (const T&) degrees
  * @return T
  */
 template <typename T> 
 inline T Math3D::degrees_to_radians( const T& degrees )
 {
    return (degrees * PI) / ONE_EIGHTY;
 }
 
 /**
  * get_readjusted_angle_degrees() returns the angle in degrees by
  * converting the angle to be between 0 (inclusive) and 360 (exclusive).
  *
  * @param (const T&) degrees
  * @return T
  */
 template <typename T>
 inline T Math3D::get_readjusted_angle_degrees( const T& degrees )
 {
    // Modulus method
    if ( degrees >= ZERO && degrees < 360.0f )
        return degrees;
    
    long integer_degrees( degrees );
    T degrees_diff( degrees - integer_degrees );
    T readjusted_degrees( (integer_degrees % 360) + degrees_diff );
    
    if ( readjusted_degrees < ZERO )
        readjusted_degrees += 360.0f;
    
    return readjusted_degrees;
    
    // Less accurate than the above, especially for large degrees value.
    // However, the algorithm below is not necessarily slower.
    
    //if ( degrees >= ZERO && degrees < 360.0f )
    //    return degrees;
    
    //T radian( Math3D::degrees_to_radians( degrees ) );
    
    //return Vector::get_angle_degrees( -sin( radian ), cos( radian ) );
 }
 
 /**
  * radians_to_degrees() returns the degrees given the radians.
  *
  * @param (const T&) radians
  * @return T
  */
 template <typename T> 
 inline T Math3D::radians_to_degrees( const T& radians )
 {
    return (radians * ONE_EIGHTY) / PI;
 }

 #endif // MATH3D_H